package gov.va.med.domain.service.messaging.encode.hl7;

import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.model.User;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7HeaderParameters;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7v23Helper;
import gov.va.med.domain.service.messaging.encode.hl7.util.MPIFieldHelper;
import gov.va.med.domain.service.messaging.environment.EndPoint;
import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.v23.datatype.QSC;
import ca.uhn.hl7v2.model.v23.datatype.RCD;
import ca.uhn.hl7v2.model.v23.message.VQQ_Q02;
import ca.uhn.hl7v2.model.v23.segment.RDF;
import ca.uhn.hl7v2.model.v23.segment.VTQ;
import ca.uhn.hl7v2.parser.Parser;
import ca.uhn.hl7v2.parser.PipeParser;

/**
 * MpiQueryEncoder
 * Encode MPI query message with First Name, Last Name, SSN, Date of Birth, using
 * the VQQ_Q02 message structure.
 *
 * @author Slava Uchitel
 * @version $Id: MpiQueryEncoder.java,v 1.5 2005/07/22 20:59:10 tom Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public class MpiQueryEncoder extends HL7MessageEncoder {
	private static String EVENT = "VQQ";
	private static String TRIGGER = "Q02";
	private static String MPI_CONJUNCTION_AND = "AND";
	private static String MPI_RELATION_EQUALS = "EQ";
	private static String SENDING_APP = "MPI_LOAD";
	private static String RECEIVING_APP = "MPI-ICN";
	private static String QUERY_FORMAT_CODE = "T";
	private static String QUERY_NAME = "VTQ_PID_ICN_NO_LOAD";
	private static String VIRTUAL_TABLE_NAME = "ICN";

	/**
	 * Build a VQQ_Q02 message with MSH, VTD and RDF segments
	 */
	public Object encode(IPayload payload, EndPoint endPoint)
	    throws MessagingException {
		User user = (User)payload;
		VQQ_Q02 vqq_q02 = null;

		//instantiate a PipeParser, which handles the "traditional encoding"
		Parser parser = new PipeParser();

		//construct message
		try {
			vqq_q02 = createQueryMessage(user, buildHL7HeaderParameters(endPoint));
			return parser.encode(vqq_q02);
		}
		catch(DataTypeException e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
		catch(HL7Exception e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
	}

	/*
	String mpiQuery =
		"MSH|^~\\&|MPI_LOAD|200M|MPI-ICN||20050202105053-0500||VQQ^Q02|1107359453570106476|T|2.3\r"+
	    "VTQ|432786934|T|VTQ_PID_ICN_NO_LOAD|ICN|@00108.1^EQ^LOWERY^AND~@00108.2^EQ^RANDY^AND~@00122^EQ^432786934^AND~@00110^EQ^19430914\r"+
	    "RDF|8|@00108.2^ST^30~@00108.3^ST^16~@00108.1^ST^30~@00122^ST^9~@00110^ST^8~@00105^ST^19~@00756^ST^5~@00169^ST^999\r";
	*/
    
	private VQQ_Q02 createQueryMessage(User user, HL7HeaderParameters params)
	    throws HL7Exception, DataTypeException {
		VQQ_Q02 vqq_q02 = new VQQ_Q02();
		//MSH
		HL7v23Helper.fillMSHSegment(vqq_q02.getMSH(), params);
		//VTD
		buildVtqSegment(user, vqq_q02);
		//RDF
		buildRdfSegment(vqq_q02);
		return vqq_q02;
	}

	private void buildRdfSegment(VQQ_Q02 vqq_q02) throws DataTypeException, HL7Exception {

		RDF rdf = vqq_q02.getRDF();
		rdf.getNumberOfColumnsPerRow().setValue(String.valueOf(MPIFieldHelper.fieldDefs.length));
		for(int i = 0; i < MPIFieldHelper.fieldDefs.length; i++) {
			RCD rcd = rdf.getColumnDescription(i);
			rcd.getHL7ItemNumber().setValue(MPIFieldHelper.fieldDefs[i][0]);
			rcd.getHL7DateType().setValue(MPIFieldHelper.fieldDefs[i][1]);
			rcd.getMaximumColumnWidth().setValue(MPIFieldHelper.fieldDefs[i][2]);
		}
	}

	private void buildVtqSegment(User user, VQQ_Q02 vqq_q02) throws DataTypeException, HL7Exception {
		//VTQ
		VTQ vtq = vqq_q02.getVTQ();
		vtq.getQueryTag().setValue(user.getSocialSecurityNumber());
		vtq.getQueryResponseFormatCode().setValue(QUERY_FORMAT_CODE);
		vtq.getVTQueryName().getIdentifier().setValue(QUERY_NAME);
		vtq.getVirtualTableName().getIdentifier().setValue(VIRTUAL_TABLE_NAME);

		buildQscCtriterion(user, vtq);
	}

	private void buildQscCtriterion(User user, VTQ vtq) throws HL7Exception, DataTypeException {

		QSC c0 = vtq.getSelectionCriteria(0);
		c0.getNameOfField().setValue(MPIFieldHelper.MPI_LAST_NAME);
		c0.getRelationalOperator().setValue(MPI_RELATION_EQUALS);
		c0.getRelationalConjunction().setValue(MPI_CONJUNCTION_AND);
		c0.getValue().setValue(user.getLastName());

		QSC c1 = vtq.getSelectionCriteria(1);
		c1.getNameOfField().setValue(MPIFieldHelper.MPI_FIRST_NAME);
		c1.getRelationalOperator().setValue(MPI_RELATION_EQUALS);
		c1.getRelationalConjunction().setValue(MPI_CONJUNCTION_AND);
		c1.getValue().setValue(user.getFirstName());

		QSC c2 = vtq.getSelectionCriteria(2);
		c2.getNameOfField().setValue(MPIFieldHelper.MPI_SSN);
		c2.getRelationalOperator().setValue(MPI_RELATION_EQUALS);
		c2.getRelationalConjunction().setValue(MPI_CONJUNCTION_AND);
		c2.getValue().setValue(user.getSocialSecurityNumber());
		String birthDate = dateToyyyyMMdd(user.getDateOfBirth());

		QSC c3 = vtq.getSelectionCriteria(3);
		c3.getNameOfField().setValue(MPIFieldHelper.MPI_DOB);
		c3.getRelationalOperator().setValue(MPI_RELATION_EQUALS);
		c3.getValue().setValue(birthDate);
	}

	private HL7HeaderParameters buildHL7HeaderParameters(EndPoint endPoint) {
		HL7HeaderParameters params = new HL7HeaderParameters(endPoint);
		params.setDelimiters(DEFAULT_DELIMITERS);
		params.setFieldSeparator(DEFAULT_FIELD_SEPARATOR);
		params.setEvent(EVENT);
		params.setTrigger(TRIGGER);
		params.setSrcStationNumber(endPoint.getEncodingParameter(RECEIVING_FACILITY));
		params.setSrcDomain(EMPTY_STRING);
		params.setSrcApplication(SENDING_APP);
		params.setDstStationNumber(EMPTY_STRING);
		params.setDstDomain(EMPTY_STRING);
		params.setDstApplication(RECEIVING_APP);
		params.setMessageControlId(EMPTY_STRING);
		params.setAcceptAcknowledgmentType(EMPTY_STRING);
		params.setApplicationAcknowledgmentType(EMPTY_STRING);
		params.setAcceptAcknowledgmentType(EMPTY_STRING);
		return params;
	}

}
